<?php
// ✅ 允许跨域
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') exit;

date_default_timezone_set('Asia/Shanghai');

define('DATA_DIR', __DIR__ . '/data');
define('TEXT_DIR', DATA_DIR . '/texts');
define('META_DIR', DATA_DIR . '/meta');
define('FILE_DIR', DATA_DIR . '/files');
define('FILE_META_DIR', DATA_DIR . '/file_meta');
define('TRUSTED_FILE', DATA_DIR . '/trusted_browsers.json');

@mkdir(DATA_DIR, 0755, true);
@mkdir(TEXT_DIR, 0755, true);
@mkdir(META_DIR, 0755, true);
@mkdir(FILE_DIR, 0755, true);
@mkdir(FILE_META_DIR, 0755, true);

// 安全加载文件操作密码
function getFilePassword() {
    $passFile = DATA_DIR . '/.password';
    if (!file_exists($passFile)) {
        file_put_contents($passFile, "mima11");
        chmod($passFile, 0600);
    }
    return trim(file_get_contents($passFile));
}

// 安全清理 ID
function sanitizeId($id) {
    $id = trim($id);
    if ($id === '') return '';
    return preg_replace('/[^a-zA-Z0-9_\-]/', '', $id);
}

function safeBasename($path) {
    if (function_exists('mb_substr')) {
        $pos = mb_strrpos($path, '/');
        return $pos === false ? $path : mb_substr($path, $pos + 1);
    }
    return basename($path);
}

function getBrowserFingerprint() {
    $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
    return hash('sha256', $ip . $ua . 'SALT2025');
}

function isTrustedBrowser() {
    if (!file_exists(TRUSTED_FILE)) return false;
    $trusted = json_decode(file_get_contents(TRUSTED_FILE), true) ?: [];
    $fp = getBrowserFingerprint();
    $now = time();
    $cleaned = [];
    foreach ($trusted as $f => $ts) {
        if ($now - $ts < 86400) {
            $cleaned[$f] = $ts;
        }
    }
    if ($cleaned !== $trusted) {
        file_put_contents(TRUSTED_FILE, json_encode($cleaned));
    }
    return isset($cleaned[$fp]);
}

function markAsTrusted() {
    $trusted = json_decode(file_get_contents(TRUSTED_FILE) ?: '[]', true) ?: [];
    $fp = getBrowserFingerprint();
    $trusted[$fp] = time();
    file_put_contents(TRUSTED_FILE, json_encode($trusted));
}

// === 新增：密码验证接口 ===
if ($_GET['action'] === 'verify_password') {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        exit;
    }
    $inputPass = $_POST['password'] ?? '';
    $correctPass = getFilePassword();
    if ($inputPass === $correctPass) {
        markAsTrusted();
        echo json_encode(['status' => 'ok']);
    } else {
        http_response_code(403);
        echo json_encode(['error' => '密码错误']);
    }
    exit;
}

// === 清理过期文件 ===
function cleanupExpiredFiles() {
    $now = time();
    foreach (glob(FILE_DIR . '/*') as $filePath) {
        $filename = basename($filePath);
        $metaFile = FILE_META_DIR . '/' . $filename . '.json';
        $expireAt = null;

        if (file_exists($metaFile)) {
            $meta = json_decode(file_get_contents($metaFile), true);
            if (isset($meta['expire_days'])) {
                if ($meta['expire_days'] == 999) continue;
                $expireAt = filemtime($filePath) + $meta['expire_days'] * 86400;
            }
        }

        if ($expireAt !== null && $now > $expireAt) {
            unlink($filePath);
            if (file_exists($metaFile)) unlink($metaFile);
        }
    }
}

// === 查看文件内容（文本类）===
if (isset($_GET['raw_file'])) {
    $filename = preg_replace('/[^\p{L}\p{N}\.\-_]/u', '_', $_GET['raw_file'] ?? '');
    $filename = trim($filename, '._');
    $path = FILE_DIR . '/' . $filename;

    if (!$filename || !file_exists($path)) {
        http_response_code(404);
        exit('文件不存在');
    }

    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    $textTypes = ['txt', 'json', 'yaml', 'yml', 'sh', 'js', 'css', 'html', 'xml', 'ini', 'log', 'md', 'conf', 'csv'];
    if (!in_array($ext, $textTypes)) {
        http_response_code(403);
        exit('仅支持文本文件查看');
    }

    header('Content-Type: text/plain; charset=utf-8');
    readfile($path);
    exit;
}

if (isset($_GET['raw'])) {
    $id = sanitizeId($_GET['raw']);
    if (!$id) {
        http_response_code(400);
        exit('ID 无效');
    }
    $textFile = TEXT_DIR . "/$id.txt";
    $metaFile = META_DIR . "/$id.json";

    if (!file_exists($textFile)) {
        http_response_code(404);
        exit('文本不存在');
    }

    if (file_exists($metaFile)) {
        $meta = json_decode(file_get_contents($metaFile), true);
        if (!empty($meta['password_hash'])) {
            $inputPass = $_GET['password'] ?? '';
            if (hash('sha256', $inputPass) !== $meta['password_hash']) {
                http_response_code(403);
                exit('密码错误');
            }
        }
    }

    header('Content-Type: text/plain; charset=utf-8');
    readfile($textFile);
    exit;
}

if (isset($_GET['get_text_content'])) {
    $id = sanitizeId($_GET['id']);
    if (!$id || !file_exists(TEXT_DIR . "/$id.txt")) {
        http_response_code(404);
        echo json_encode(['error' => '文本不存在']);
        exit;
    }

    $metaFile = META_DIR . "/$id.json";
    $hasPassword = false;
    $passwordHash = '';

    if (file_exists($metaFile)) {
        $meta = json_decode(file_get_contents($metaFile), true);
        if (!empty($meta['password_hash'])) {
            $hasPassword = true;
            $passwordHash = $meta['password_hash'];
        }
    }

    if ($hasPassword) {
        $inputPass = $_GET['password'] ?? '';
        if (hash('sha256', $inputPass) !== $passwordHash) {
            http_response_code(403);
            echo json_encode(['error' => '密码错误']);
            exit;
        }
    }

    $content = file_get_contents(TEXT_DIR . "/$id.txt");
    echo json_encode([
        'content' => $content,
        'hasPassword' => $hasPassword
    ]);
    exit;
}

// === API 处理 ===
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($_GET['action'] === 'save_text') {
        $id = sanitizeId($_POST['id'] ?? '');
        $content = $_POST['content'] ?? '';
        $password = $_POST['password'] ?? '';

        if (!trim($content)) {
            http_response_code(400);
            echo json_encode(['error' => '内容不能为空']);
            exit;
        }

        if (strlen($content) > 10 * 1024 * 1024) {
            http_response_code(400);
            echo json_encode(['error' => '文本内容过大（最大 10MB）']);
            exit;
        }

        $isUpdate = $id !== '' && file_exists(TEXT_DIR . "/$id.txt");

        if ($id === '') {
            $id = 'note_' . substr(uniqid(), -6);
        }

        if (!$isUpdate && file_exists(TEXT_DIR . "/$id.txt")) {
            http_response_code(409);
            echo json_encode(['error' => 'ID 已存在，请换一个']);
            exit;
        }

        file_put_contents(TEXT_DIR . "/$id.txt", $content);

        $meta = [];
        if ($isUpdate && file_exists(META_DIR . "/$id.json")) {
            $oldMeta = json_decode(file_get_contents(META_DIR . "/$id.json"), true);
            $oldPasswordHash = $oldMeta['password_hash'] ?? '';
        } else {
            $oldPasswordHash = '';
        }

        if ($password === '') {
            if ($oldPasswordHash !== '') {
                $meta['password_hash'] = $oldPasswordHash;
            }
        } else {
            $meta['password_hash'] = hash('sha256', $password);
        }

        file_put_contents(META_DIR . "/$id.json", json_encode($meta, JSON_UNESCAPED_UNICODE));
        echo json_encode(['status' => 'ok', 'id' => $id, 'isUpdate' => $isUpdate]);
        exit;
    } elseif ($_GET['action'] === 'upload_file') {
        // 此时应已通过 verify_password 验证，或为 trusted
        if (!isTrustedBrowser()) {
            http_response_code(403);
            echo json_encode(['error' => '未授权']);
            exit;
        }

        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['error' => '❌ 上传失败']);
            exit;
        }

        $originalName = $_FILES['file']['name'];
        $safeName = preg_replace('/[^\p{L}\p{N}\.\-_]/u', '_', $originalName);
        $safeName = trim($safeName, '._') ?: 'file_' . substr(uniqid(), -6);

        if (file_exists(FILE_DIR . '/' . $safeName)) {
            http_response_code(409);
            echo json_encode(['error' => '文件名已存在']);
            exit;
        }

        if (!move_uploaded_file($_FILES['file']['tmp_name'], FILE_DIR . '/' . $safeName)) {
            http_response_code(500);
            echo json_encode(['error' => '❌ 保存失败']);
            exit;
        }

        $expireDays = intval($_POST['expire_days'] ?? 999);
        if ($expireDays < 1 || $expireDays > 999) $expireDays = 999;
        file_put_contents(FILE_META_DIR . '/' . $safeName . '.json', json_encode(['expire_days' => $expireDays]));
        echo json_encode(['status' => 'ok']);
        exit;
    }
} elseif ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    if ($_GET['action'] === 'delete_text') {
        $id = sanitizeId($_GET['id'] ?? '');
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'ID 无效']);
            exit;
        }
        $textFile = TEXT_DIR . "/$id.txt";
        $metaFile = META_DIR . "/$id.json";

        if (!file_exists($textFile)) {
            http_response_code(404);
            echo json_encode(['error' => '文本不存在']);
            exit;
        }

        if (file_exists($metaFile)) {
            $meta = json_decode(file_get_contents($metaFile), true);
            if (!empty($meta['password_hash'])) {
                $inputPass = $_GET['password'] ?? '';
                if (hash('sha256', $inputPass) !== $meta['password_hash']) {
                    http_response_code(403);
                    echo json_encode(['error' => '密码错误']);
                    exit;
                }
            }
        }

        unlink($textFile);
        if (file_exists($metaFile)) unlink($metaFile);
        echo json_encode(['status' => 'ok']);
        exit;
    } elseif ($_GET['action'] === 'delete_file') {
        if (!isTrustedBrowser()) {
            http_response_code(403);
            echo json_encode(['error' => '未授权']);
            exit;
        }

        $filename = preg_replace('/[^\p{L}\p{N}\.\-_]/u', '_', $_GET['filename'] ?? '');
        $filename = trim($filename, '._');
        $path = FILE_DIR . '/' . $filename;
        $metaPath = FILE_META_DIR . '/' . $filename . '.json';

        if ($filename && file_exists($path)) {
            unlink($path);
            if (file_exists($metaPath)) unlink($metaPath);
            echo json_encode(['status' => 'ok']);
        } else {
            http_response_code(404);
            echo json_encode(['error' => '文件不存在']);
        }
        exit;
    }
}

// === 文件管理页面 ===
if (($_GET['page'] ?? '') === 'files') {
    cleanupExpiredFiles();

    $files = [];
    foreach (glob(FILE_DIR . '/*') as $f) {
        $name = safeBasename($f);
        $mtime = filemtime($f);
        $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
        $isText = in_array($ext, ['txt','json','yaml','yml','sh','js','css','html','xml','ini','log','md','conf','csv']);
        $files[] = [
            'name' => $name,
            'time' => date('Y-m-d H:i:s', $mtime),
            'mtime' => $mtime,
            'isText' => $isText
        ];
    }

    usort($files, fn($a, $b) => $b['mtime'] <=> $a['mtime']);
    $files = array_slice($files, 0, 100);
    ?>
    <!DOCTYPE html>
    <html lang="zh">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <title>📁 文件管理 - ShareIt</title>
        <style>
            :root { --primary: #4361ee; --danger: #e63946; --success: #4cc9f0; }
            * { box-sizing: border-box; }
            body { font-family: system-ui; margin: 0; padding: 16px; background: #fafafa; }
            .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
            .btn { padding: 10px 16px; border: none; border-radius: 6px; cursor: pointer; font-weight: bold; text-decoration: none; display: inline-block; }
            .primary { background: var(--primary); color: white; }
            .back { color: var(--primary); }
            .item { display: flex; justify-content: space-between; padding: 12px; background: white; margin: 8px 0; border-radius: 8px; }
            .file-info { flex: 1; }
            .file-name { font-weight: bold; margin-bottom: 4px; }
            .file-time { font-size: 0.85em; color: #666; }
            input, button { margin: 4px 0; padding: 8px 12px; border: 1px solid #ccc; border-radius: 4px; }
            button.danger { background: var(--danger); color: white; border: none; }
            #expireInput { display: none; }
            .text-link { color: var(--primary); text-decoration: underline; margin-right: 8px; }
            label { display: block; margin: 6px 0 4px; font-weight: bold; }
            #progressContainer { margin: 10px 0; display: none; }
            #progressBar { width: 100%; height: 12px; background: #e0e0e0; border-radius: 6px; overflow: hidden; }
            #progress { height: 100%; background: var(--success); width: 0%; transition: width 0.2s; }
            #progressText { text-align: center; font-size: 0.9em; margin-top: 4px; color: #555; }
        </style>
    </head>
    <body>
        <div class="header">
            <h2>📁 文件管理</h2>
            <a href="/" class="back">← 返回文本</a>
        </div>

        <form id="uploadForm" enctype="multipart/form-data">
            <input type="file" name="file" required>
            <label for="expireInput">有效期（天）: 1-998，999=永久</label>
            <input type="number" id="expireInput" name="expire_days" min="1" max="999" value="999">
            <button type="submit" class="primary">📤 上传</button>
        </form>

        <div id="progressContainer">
            <div id="progressBar"><div id="progress"></div></div>
            <div id="progressText">0%</div>
        </div>

        <div id="fileList">
            <?php if (empty($files)): ?>
                <p>暂无文件</p>
            <?php else: ?>
                <?php foreach ($files as $file): ?>
                    <div class="item">
                        <div class="file-info">
                            <?php if ($file['isText']): ?>
                                <div class="file-name">
                                    <a href="?raw_file=<?= rawurlencode($file['name']) ?>" target="_blank"><?= htmlspecialchars($file['name']) ?></a>
                                </div>
                            <?php else: ?>
                                <div class="file-name"><?= htmlspecialchars($file['name']) ?></div>
                            <?php endif; ?>
                            <div class="file-time"><?= htmlspecialchars($file['time']) ?></div>
                        </div>
                        <div>
                            <a href="data/files/<?= rawurlencode($file['name']) ?>" download>📥</a>
                            <button class="danger" onclick="del('<?= addslashes($file['name']) ?>')">🗑️</button>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <script>
            let isTrusted = false;

            // 检查是否已信任
            async function checkTrusted() {
                try {
                    const res = await fetch('?action=delete_file&filename=TEST&password=', { method: 'DELETE' });
                    isTrusted = (res.status === 404); // 404 = 通过认证（文件不存在）
                    if (isTrusted) {
                        document.getElementById('expireInput').style.display = 'block';
                    }
                    return isTrusted;
                } catch (e) {
                    return false;
                }
            }

            // 验证密码
            async function verifyPassword(pass) {
                const res = await fetch('?action=verify_password', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: 'password=' + encodeURIComponent(pass)
                });
                if (res.ok) {
                    isTrusted = true;
                    document.getElementById('expireInput').style.display = 'block';
                    return true;
                } else {
                    const err = await res.json();
                    alert('❌ ' + (err.error || '密码错误'));
                    return false;
                }
            }

            // 上传前验证
            document.getElementById('uploadForm').addEventListener('submit', async (e) => {
                e.preventDefault();
                if (!isTrusted) {
                    const pass = prompt('🔒 请输入操作密码：');
                    if (pass === null) return;
                    const ok = await verifyPassword(pass);
                    if (!ok) return;
                }

                const form = new FormData(e.target);
                const xhr = new XMLHttpRequest();
                const progressContainer = document.getElementById('progressContainer');
                const progressBar = document.getElementById('progress');
                const progressText = document.getElementById('progressText');

                xhr.upload.onprogress = function(event) {
                    if (event.lengthComputable) {
                        const percent = Math.round((event.loaded / event.total) * 100);
                        progressBar.style.width = percent + '%';
                        progressText.textContent = percent + '%';
                    }
                };

                xhr.onload = function() {
                    progressContainer.style.display = 'none';
                    if (xhr.status >= 200 && xhr.status < 300) {
                        const d = JSON.parse(xhr.responseText);
                        if (d.status === 'ok') {
                            location.reload();
                        } else {
                            alert('❌ ' + (d.error || '上传失败'));
                        }
                    } else {
                        alert('❌ 上传失败: ' + xhr.status);
                    }
                };

                xhr.onerror = function() {
                    progressContainer.style.display = 'none';
                    alert('❌ 网络错误');
                };

                xhr.open('POST', '?action=upload_file');
                progressContainer.style.display = 'block';
                progressBar.style.width = '0%';
                progressText.textContent = '0%';
                xhr.send(form);
            });

            // 删除前验证
            async function del(name) {
                if (!isTrusted) {
                    const pass = prompt('🔒 请输入操作密码：');
                    if (pass === null) return;
                    const ok = await verifyPassword(pass);
                    if (!ok) return;
                }
                const url = `?action=delete_file&filename=${encodeURIComponent(name)}`;
                const res = await fetch(url, { method: 'DELETE' });
                if (res.ok) {
                    location.reload();
                } else {
                    const err = await res.json();
                    alert('❌ ' + (err.error || '删除失败'));
                }
            }

            // 初始化：检查是否已信任
            checkTrusted();
        </script>
    </body>
    </html>
    <?php
    exit;
}

// === 主界面 ===
$texts = [];
foreach (glob(TEXT_DIR . '/*.txt') as $f) {
    $id = pathinfo($f, PATHINFO_FILENAME);
    $mtime = filemtime($f);
    $metaFile = META_DIR . "/$id.json";
    $encrypted = file_exists($metaFile) && !empty(json_decode(file_get_contents($metaFile), true)['password_hash']);
    $texts[] = ['id' => $id, 'encrypted' => $encrypted, 'mtime' => $mtime];
}

usort($texts, fn($a, $b) => $b['mtime'] <=> $a['mtime']);
$texts = array_slice($texts, 0, 100);
?>
<!DOCTYPE html>
<html lang="zh">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>📝 文本分享 - ShareIt</title>
    <style>
        :root { --primary: #4361ee; --danger: #e63946; }
        * { box-sizing: border-box; }
        body { font-family: system-ui; margin: 0; padding: 16px; background: #fafafa; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
        .btn { padding: 10px 16px; border: none; border-radius: 6px; cursor: pointer; font-weight: bold; text-decoration: none; display: inline-block; }
        .primary { background: var(--primary); color: white; }
        textarea { width: 100%; height: 160px; padding: 12px; margin: 8px 0; border: 1px solid #ddd; border-radius: 8px; font-family: monospace; resize: vertical; }
        .list { margin-top: 24px; }
        .item { display: flex; justify-content: space-between; padding: 12px; background: white; margin: 8px 0; border-radius: 8px; align-items: center; }
        .name-area { flex: 1; display: flex; align-items: center; gap: 8px; }
        .name-link { color: var(--primary); font-weight: bold; text-decoration: none; cursor: pointer; }
        .raw-link { color: #666; font-size: 0.9em; text-decoration: underline; }
        .encrypted::after { content: " 🔒"; }
        button.danger { background: var(--danger); color: white; border: none; padding: 6px 10px; border-radius: 4px; }
        input { width: 100%; padding: 8px; margin: 4px 0; border: 1px solid #ccc; border-radius: 4px; }
        label { display: block; margin-top: 8px; font-weight: bold; }
    </style>
</head>
<body>
    <div class="header">
        <h1>📝 文本分享</h1>
        <a href="?page=files" class="btn primary">📁 文件管理</a>
    </div>

    <form id="textForm">
        <label for="textId">文本 ID（留空则自动生成）:</label>
        <input type="text" id="textId" placeholder="例如: my-note">
        <textarea id="content" placeholder="输入文本内容..."></textarea>
        <input type="text" id="password" placeholder="">
        <button type="submit" class="btn primary">💾 保存文本</button>
    </form>

    <div class="list" id="textList">
        <?php if (empty($texts)): ?>
            <p>暂无文本，创建一个吧！</p>
        <?php else: ?>
            <?php foreach ($texts as $t): ?>
                <div class="item" data-id="<?= htmlspecialchars($t['id']) ?>">
                    <div class="name-area">
                        <span class="name-link" onclick="editText('<?= addslashes($t['id']) ?>', <?= $t['encrypted'] ? 'true' : 'false' ?>)" title="点击编辑">
                            <?= htmlspecialchars($t['id']) ?><?= $t['encrypted'] ? ' 🔒' : '' ?>
                        </span>
                        <a href="?raw=<?= urlencode($t['id']) ?>" target="_blank" class="raw-link">📄 源文本</a>
                    </div>
                    <button class="danger" onclick="delText('<?= addslashes($t['id']) ?>', <?= $t['encrypted'] ? 'true' : 'false' ?>)">🗑️</button>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>

    <script>
        async function editText(id, encrypted) {
            let password = '';
            if (encrypted) {
                password = prompt('🔒 此文本已加密，请输入密码：');
                if (password === null) return;
            }

            const url = `?get_text_content&id=${encodeURIComponent(id)}` + 
                        (password ? `&password=${encodeURIComponent(password)}` : '');
            const res = await fetch(url);
            if (!res.ok) {
                const err = await res.json();
                alert('❌ ' + (err.error || '加载失败'));
                return;
            }
            const data = await res.json();
            document.getElementById('textId').value = id;
            document.getElementById('content').value = data.content;
            document.getElementById('password').value = '';
            document.getElementById('password').placeholder = '';
            document.getElementById('textId').scrollIntoView({ behavior: 'smooth' });
        }

        document.getElementById('textForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            const id = document.getElementById('textId').value.trim();
            const content = document.getElementById('content').value.trim();
            const password = document.getElementById('password').value;
            if (!content) return alert('⚠️ 内容不能为空');

            const res = await fetch('?action=save_text', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: `id=${encodeURIComponent(id)}&content=${encodeURIComponent(content)}&password=${encodeURIComponent(password)}`
            });
            const d = await res.json();
            if (d.status === 'ok') {
                document.getElementById('textId').value = '';
                document.getElementById('content').value = '';
                document.getElementById('password').value = '';
                document.getElementById('password').placeholder = '';

                if (!d.isUpdate) {
                    const list = document.getElementById('textList');
                    const item = document.createElement('div');
                    item.className = 'item';
                    item.dataset.id = d.id;
                    item.innerHTML = `
                        <div class="name-area">
                            <span class="name-link" onclick="editText('${d.id}', false)" title="点击编辑">${d.id}</span>
                            <a href="?raw=${encodeURIComponent(d.id)}" target="_blank" class="raw-link">📄 源文本</a>
                        </div>
                        <button class="danger" onclick="delText('${d.id}', false)">🗑️</button>
                    `;
                    list.insertBefore(item, list.firstChild);
                    if (list.children.length > 100) {
                        list.removeChild(list.lastChild);
                    }
                }
            } else {
                alert('❌ ' + (d.error || '保存失败'));
            }
        });

        async function delText(id, encrypted) {
            let pass = '';
            if (encrypted) {
                pass = prompt('🔒 此文本已加密，请输入密码：');
                if (pass === null) return;
            }
            const url = `?action=delete_text&id=${encodeURIComponent(id)}` + (pass ? `&password=${encodeURIComponent(pass)}` : '');
            const res = await fetch(url, { method: 'DELETE' });
            if (res.ok) {
                const el = document.querySelector(`.item[data-id="${id}"]`);
                if (el) el.remove();
            } else {
                const err = await res.json();
                alert('❌ ' + (err.error || '删除失败'));
            }
        }
    </script>
</body>
</html>