<?php
// index.php - 笔记应用（InfinityFree 跨域兼容版）
// 作者：AI助手

// ===================== 1. 基础设置 =====================
if (isset($_SERVER['HTTP_ORIGIN'])) {
    header("Access-Control-Allow-Origin: *");
    header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
    header("Access-Control-Allow-Headers: Content-Type");
}

// 处理 OPTIONS 预检请求（必须能被外部访问到）
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

date_default_timezone_set('Asia/Shanghai');

// ===================== 2. 配置 =====================
define('NOTES_DIR', __DIR__ . '/notes/');

if (!is_dir(NOTES_DIR)) {
    mkdir(NOTES_DIR, 0755, true);
}

// ===================== 3. 获取当前笔记ID =====================
$currentId = isset($_GET['id']) ? $_GET['id'] : 'default';
if (empty($currentId)) {
    $currentId = 'default';
}
$currentId = preg_replace('/[^a-zA-Z0-9_-]/', '', $currentId);

// ===================== 4. 处理API请求（使用查询参数） =====================
if (isset($_GET['api'])) {
    $action = $_GET['api'];
    $noteId = $_GET['id'] ?? '';
    $password = $_GET['pwd'] ?? null;

    // 验证ID格式
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $noteId)) {
        http_response_code(400);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['error' => '无效的笔记ID格式']);
        exit();
    }

    $noteFile = NOTES_DIR . 'note_' . $noteId . '.txt';
    $metaFile = NOTES_DIR . 'meta_' . $noteId . '.json';

    switch ($action) {
        // ========== 获取笔记内容 ==========
        case 'raw':
            if (file_exists($metaFile)) {
                $meta = json_decode(file_get_contents($metaFile), true);
                if (isset($meta['password'])) {
                    if ($password === null) {
                        http_response_code(403);
                        echo "需要密码";
                        exit();
                    }
                    $inputHash = hash('sha256', $password);
                    if ($inputHash !== $meta['password']) {
                        http_response_code(403);
                        echo "密码错误";
                        exit();
                    }
                }
            }
            $content = file_exists($noteFile) ? file_get_contents($noteFile) : '';
            header('Content-Type: text/plain; charset=utf-8');
            echo $content;
            exit();

        // ========== 获取笔记元信息 ==========
        case 'meta':
            $meta = file_exists($metaFile) ? json_decode(file_get_contents($metaFile), true) : [];
            $response = [
                'encrypted' => isset($meta['password']),
                'format' => 'txt',
                'size' => file_exists($noteFile) ? filesize($noteFile) : 0,
                'modified' => file_exists($noteFile) ? date('Y-m-d H:i:s', filemtime($noteFile)) : null
            ];
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode($response);
            exit();

        // ========== 保存笔记（POST） ==========
        case 'save':
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                http_response_code(405);
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode(['error' => '只接受POST请求']);
                exit();
            }

            $input = file_get_contents('php://input');
            $data = json_decode($input, true);

            if ($data === null) {
                http_response_code(400);
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode(['error' => '无效的JSON数据']);
                exit();
            }

            $content = $data['content'] ?? '';
            $newPassword = $data['password'] ?? null;

            if (file_put_contents($noteFile, $content) === false) {
                http_response_code(500);
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode(['error' => '保存笔记内容失败']);
                exit();
            }

            $metaData = ['format' => 'txt'];
            if ($newPassword !== null && $newPassword !== '') {
                $metaData['password'] = hash('sha256', trim($newPassword));
            }

            if (file_put_contents($metaFile, json_encode($metaData, JSON_PRETTY_PRINT)) === false) {
                http_response_code(500);
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode(['error' => '保存元数据失败']);
                exit();
            }

            header('Content-Type: application/json; charset=utf-8');
            echo json_encode([
                'status' => 'success',
                'message' => '保存成功',
                'id' => $noteId,
                'time' => date('Y-m-d H:i:s'),
                'size' => strlen($content)
            ]);
            exit();

        default:
            http_response_code(400);
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode(['error' => '未知API操作']);
            exit();
    }
}

// ===================== 5. 显示前端页面 =====================
?>
<!DOCTYPE html>
<html lang="zh">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title><?php echo htmlspecialchars($currentId); ?> @ 我的笔记</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { 
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 0;
            margin: 0;
        }
        
        .status-bar-top {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            background: rgba(255, 255, 255, 0.95);
            padding: 6px 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            z-index: 1000;
            display: flex;
            align-items: center;
            gap: 8px;
            height: 50px;
            backdrop-filter: blur(10px);
            flex-wrap: wrap;
        }
        .note-id-badge {
            background: #667eea;
            color: white;
            padding: 4px 8px;
            border-radius: 10px;
            font-family: monospace;
            font-size: 12px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 120px;
        }
        .status-text {
            font-size: 13px;
            color: #333;
            padding: 4px 10px;
            background: #f0f0f0;
            border-radius: 12px;
            min-width: 80px;
            text-align: center;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        .status-saving { background: #ffeb3b; color: #333; }
        .status-success { background: #4caf50; color: white; }
        .status-error { background: #f44336; color: white; }
        
        .toolbar-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 2px;
        }
        .toolbar-input {
            width: 90px;
            padding: 4px 6px;
            font-size: 12px;
            border: 1px solid #ccc;
            border-radius: 6px;
            background: white;
        }
        .toolbar-btn {
            width: 32px;
            height: 32px;
            border: none;
            border-radius: 6px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            font-size: 14px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .toolbar-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 5px rgba(102,126,234,0.4);
        }
        .btn-label {
            font-size: 8px;
            color: #666;
            white-space: nowrap;
        }
        @media (max-width: 560px) {
            .btn-label { display: none; }
            .toolbar-input { width: 70px; font-size: 11px; }
            .toolbar-btn { width: 28px; height: 28px; font-size: 12px; }
        }
        
        .container {
            margin: 50px 0 0 0;
            background: white;
            height: calc(100vh - 50px);
            display: flex;
            flex-direction: column;
        }
        
        .api-display {
            padding: 6px 10px;
            background: #f8f9fa;
            border-bottom: 1px solid #e9ecef;
            font-size: 10px;
            color: #666;
            flex-shrink: 0;
        }
        .api-link {
            color: #667eea;
            text-decoration: none;
            font-family: monospace;
            background: white;
            padding: 3px 6px;
            border-radius: 4px;
            border: 1px solid #dee2e6;
            display: block;
            word-break: break-all;
        }
        .api-link:hover {
            text-decoration: underline;
            background: #f0f4ff;
        }
        
        .editor-container {
            display: flex;
            flex: 1;
            min-height: 0;
        }
        .line-numbers {
            width: 40px;
            background: #f8f9fa;
            padding: 12px 5px;
            text-align: right;
            font-family: monospace;
            color: #999;
            line-height: 1.5;
            user-select: none;
            border-right: 1px solid #e9ecef;
            font-size: 12px;
            overflow: hidden;
        }
        textarea {
            flex: 1;
            padding: 12px;
            border: none;
            font-family: monospace;
            font-size: 14px;
            line-height: 1.5;
            resize: none;
            outline: none;
            background: white;
        }
        textarea:disabled {
            background: #f8f9fa;
            color: #999;
        }
        
        .info-panel {
            padding: 6px 10px;
            background: #f8f9fa;
            border-top: 1px solid #e9ecef;
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 11px;
            color: #666;
        }
        
        .password-dialog {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            display: none;
            z-index: 1001;
            width: 90%;
            max-width: 350px;
        }
        .overlay {
            position: fixed;
            top: 0; left: 0; right: 0; bottom: 0;
            background: rgba(0,0,0,0.5);
            display: none;
            z-index: 1000;
        }
        
        ::-webkit-scrollbar { width: 5px; height: 5px; }
        ::-webkit-scrollbar-track { background: #f1f1f1; }
        ::-webkit-scrollbar-thumb { background: #c1c1c1; border-radius: 3px; }
        ::-webkit-scrollbar-thumb:hover { background: #a8a8a8; }
    </style>
</head>
<body>
    <div class="status-bar-top">
        <div class="note-id-badge" title="笔记ID: <?php echo htmlspecialchars($currentId); ?>">
            <?php echo htmlspecialchars($currentId); ?>
        </div>
        
        <div class="toolbar-item">
            <input type="text" id="jumpInput" class="toolbar-input" 
                   value="<?php echo htmlspecialchars($currentId); ?>" 
                   placeholder="ID">
            <div class="btn-label">跳转</div>
        </div>
        
        <div class="toolbar-item">
            <button class="toolbar-btn" onclick="jumpToNote()">🔗</button>
            <div class="btn-label">ID</div>
        </div>
        <div class="toolbar-item">
            <button class="toolbar-btn" onclick="showPasswordDialog()">🔒</button>
            <div class="btn-label">密码</div>
        </div>
        <div class="toolbar-item">
            <button class="toolbar-btn" onclick="newNote()">📄</button>
            <div class="btn-label">新建</div>
        </div>
        <div class="toolbar-item">
            <button class="toolbar-btn" onclick="shareNote()">📤</button>
            <div class="btn-label">分享</div>
        </div>
        <div class="toolbar-item">
            <button class="toolbar-btn" onclick="toggleFullscreen()">⛶</button>
            <div class="btn-label">全屏</div>
        </div>
        
        <div class="status-text" id="globalStatus">就绪</div>
    </div>
    
    <div class="container">
        <div class="api-display">
            <a href="#" id="rawUrl" class="api-link" target="_blank">加载中...</a>
        </div>
        
        <div class="editor-container">
            <div class="line-numbers" id="lineNumbers">1</div>
            <textarea id="editor" placeholder="开始输入... 修改后1秒自动保存"></textarea>
        </div>
        
        <div class="info-panel">
            <span>📝 <span id="charCount">0</span> 字符 | 📊 <span id="lineCount">1</span> 行</span>
            <span>最后保存: <span id="lastSaveTime">-</span></span>
        </div>
    </div>
    
    <div class="overlay" id="overlay" onclick="hidePasswordDialog()"></div>
    <div class="password-dialog" id="passwordDialog">
        <h3 style="margin:0 0 10px; font-size:16px;">🔒 设置密码保护</h3>
        <p style="margin:0 0 10px; font-size:13px;">输入密码（留空则移除保护）：</p>
        <input type="password" id="passwordInput" style="width:100%; padding:8px; margin-bottom:15px; border:1px solid #ddd; border-radius:6px;">
        <div style="display:flex; gap:8px; justify-content:flex-end;">
            <button onclick="savePassword()" style="padding:6px 16px; background:linear-gradient(135deg, #667eea, #764ba2); color:white; border:none; border-radius:6px;">确定</button>
            <button onclick="hidePasswordDialog()" style="padding:6px 16px; background:#f0f0f0; border:none; border-radius:6px;">取消</button>
        </div>
    </div>
    
    <script>
        const currentId = '<?php echo $currentId; ?>';
        let isAuthorized = false;
        let saveTimer = null;
        let isSaving = false;
        let lastSavedTimeStr = '-';
        
        const baseUrl = window.location.pathname; // e.g. /index.php
        const editor = document.getElementById('editor');
        const lineNumbers = document.getElementById('lineNumbers');
        const charCount = document.getElementById('charCount');
        const lineCount = document.getElementById('lineCount');
        const rawUrl = document.getElementById('rawUrl');
        const globalStatus = document.getElementById('globalStatus');
        const passwordDialog = document.getElementById('passwordDialog');
        const overlay = document.getElementById('overlay');
        const jumpInput = document.getElementById('jumpInput');
        const lastSaveTimeEl = document.getElementById('lastSaveTime');
        
        document.addEventListener('DOMContentLoaded', function() {
            updateApiUrl();
            loadNote();
            editor.addEventListener('input', () => {
                updateLineNumbers();
                updateCharCount();
                autoSave();
            });
            editor.addEventListener('scroll', () => lineNumbers.scrollTop = editor.scrollTop);
            jumpInput.addEventListener('keypress', e => { if (e.key === 'Enter') jumpToNote(); });
            if ('ontouchstart' in window) editor.style.paddingBottom = '70px';
        });
        
        function updateLineNumbers() {
            const lines = editor.value.split('\n').length;
            lineNumbers.textContent = Array.from({length: lines}, (_, i) => i + 1).join('\n');
            lineCount.textContent = lines;
        }
        
        function updateCharCount() {
            charCount.textContent = editor.value.length;
        }
        
        function updateLastSaveTime() {
            const now = new Date();
            lastSavedTimeStr = now.toLocaleTimeString('zh-CN', { hour12: false, hour: '2-digit', minute: '2-digit', second: '2-digit' });
            lastSaveTimeEl.textContent = lastSavedTimeStr;
        }
        
        function updateApiUrl() {
            // ✅ 使用查询参数格式，兼容 InfinityFree
            const fullUrl = window.location.origin + baseUrl + '?api=raw&id=' + encodeURIComponent(currentId);
            rawUrl.href = fullUrl;
            rawUrl.textContent = fullUrl;
        }
        
        function getApiBaseUrl() {
            return window.location.origin + baseUrl + '?';
        }
        
        function setStatus(message, type = 'info') {
            let displayMessage = message;
            if (type === 'success' && message === '保存成功') {
                displayMessage = '保存成功 (' + lastSavedTimeStr + ')';
            }
            globalStatus.textContent = displayMessage;
            globalStatus.className = 'status-text status-' + (type === 'info' ? 'saving' : type);
        }
        
        async function loadNote() {
            try {
                setStatus('加载中...', 'saving');
                const metaUrl = getApiBaseUrl() + 'api=meta&id=' + encodeURIComponent(currentId);
                const metaRes = await fetch(metaUrl);
                
                if (!metaRes.ok) {
                    editor.value = '# 新笔记\n\n欢迎使用在线笔记！';
                    isAuthorized = true;
                    setStatus('新笔记', 'success');
                } else {
                    const meta = await metaRes.json();
                    let contentUrl = getApiBaseUrl() + 'api=raw&id=' + encodeURIComponent(currentId);
                    if (meta.encrypted) {
                        const pwd = prompt('🔒 输入密码：');
                        if (!pwd) {
                            editor.value = '需要密码';
                            editor.disabled = true;
                            setStatus('需要密码', 'error');
                            return;
                        }
                        contentUrl += '&pwd=' + encodeURIComponent(pwd);
                    }
                    const contentRes = await fetch(contentUrl);
                    if (contentRes.ok) {
                        editor.value = await contentRes.text();
                        isAuthorized = true;
                        setStatus(meta.encrypted ? '已加载（加密）' : '已加载', 'success');
                    } else {
                        editor.value = '加载失败';
                        editor.disabled = true;
                        setStatus('内容加载失败', 'error');
                        return;
                    }
                }
                updateLineNumbers();
                updateCharCount();
                updateLastSaveTime();
            } catch (e) {
                setStatus('加载异常', 'error');
                editor.value = '错误: ' + e.message;
            }
        }
        
        function autoSave() {
            if (!isAuthorized || isSaving) return;
            clearTimeout(saveTimer);
            saveTimer = setTimeout(saveNote, 1000);
        }
        
        async function saveNote(password = null) {
            if (!isAuthorized || isSaving) return;
            try {
                isSaving = true;
                setStatus('保存中...', 'saving');
                const saveUrl = getApiBaseUrl() + 'api=save&id=' + encodeURIComponent(currentId);
                const body = { content: editor.value };
                if (password !== null) body.password = password;
                
                const res = await fetch(saveUrl, {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(body)
                });
                
                if (res.ok) {
                    setStatus('保存成功', 'success');
                    updateLastSaveTime();
                    if (password !== null) updateApiUrl();
                } else {
                    const err = await res.json().catch(() => ({}));
                    setStatus('失败: ' + (err.error || res.statusText), 'error');
                }
            } catch (e) {
                setStatus('异常: ' + e.message, 'error');
            } finally {
                isSaving = false;
            }
        }
        
        function jumpToNote() {
            const id = jumpInput.value.trim();
            if (!id) return setStatus('请输入ID', 'error');
            if (id === currentId) return setStatus('已是当前', 'info');
            window.location.href = baseUrl + '?id=' + encodeURIComponent(id);
        }
        
        function newNote() {
            const id = 'note-' + Date.now();
            window.location.href = baseUrl + '?id=' + id;
        }
        
        function shareNote() {
            if (navigator.clipboard) {
                navigator.clipboard.writeText(window.location.href).then(() => {
                    setStatus('链接已复制', 'success');
                    setTimeout(() => setStatus('就绪'), 1500);
                });
            } else {
                prompt('复制链接：', window.location.href);
            }
        }
        
        function toggleFullscreen() {
            if (!document.fullscreenElement) {
                document.documentElement.requestFullscreen?.();
                setStatus('全屏开启', 'success');
            } else {
                document.exitFullscreen?.();
                setStatus('全屏关闭', 'success');
            }
        }
        
        function showPasswordDialog() {
            if (!isAuthorized) return alert('请先加载笔记');
            passwordDialog.style.display = 'block';
            overlay.style.display = 'block';
            setTimeout(() => document.getElementById('passwordInput').focus(), 100);
        }
        
        function hidePasswordDialog() {
            passwordDialog.style.display = 'none';
            overlay.style.display = 'none';
        }
        
        async function savePassword() {
            const pwd = document.getElementById('passwordInput').value;
            await saveNote(pwd);
            hidePasswordDialog();
            setStatus('密码已更新', 'success');
            setTimeout(() => setStatus('就绪'), 2000);
        }
        
        document.addEventListener('keydown', e => {
            if (e.ctrlKey && e.key === 's') { e.preventDefault(); saveNote(); }
            if (e.ctrlKey && e.key === 'g') { e.preventDefault(); jumpInput.select(); jumpInput.focus(); }
        });
    </script>
</body>
</html>