<?php
// index.php - 笔记应用完整版（修复网络错误）
// 作者：AI助手

// ===================== 1. 基础设置 =====================
// 允许跨域
if (isset($_SERVER['HTTP_ORIGIN'])) {
    header("Access-Control-Allow-Origin: *");
    header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
    header("Access-Control-Allow-Headers: Content-Type");
}

// 处理 OPTIONS 预检请求
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

date_default_timezone_set('Asia/Shanghai');

// ===================== 2. 配置 =====================
define('NOTES_DIR', __DIR__ . '/notes/');

if (!is_dir(NOTES_DIR)) {
    mkdir(NOTES_DIR, 0755, true);
}

// ===================== 3. 获取当前笔记ID =====================
$currentId = isset($_GET['id']) ? $_GET['id'] : 'default';
if (empty($currentId)) {
    $currentId = 'default';
}
$currentId = preg_replace('/[^a-zA-Z0-9_-]/', '', $currentId);

// ===================== 4. 处理API请求 =====================
// 获取当前脚本的目录路径
$script_path = dirname($_SERVER['SCRIPT_NAME']);
if ($script_path === '/') $script_path = '';

// 获取请求路径
$request_uri = $_SERVER['REQUEST_URI'];

// 如果是直接访问脚本（如 /index.php），重定向到干净的URL
if (strpos($request_uri, $_SERVER['SCRIPT_NAME']) === 0 && $request_uri !== $_SERVER['SCRIPT_NAME']) {
    $clean_uri = str_replace($_SERVER['SCRIPT_NAME'], '', $request_uri);
    header("Location: " . $_SERVER['SCRIPT_NAME'] . "?" . parse_url($clean_uri, PHP_URL_QUERY));
    exit();
}

// 解析请求路径
$request_path = parse_url($request_uri, PHP_URL_PATH);

// 移除脚本路径前缀（如果存在）
if ($script_path && strpos($request_path, $script_path) === 0) {
    $request_path = substr($request_path, strlen($script_path));
}

// 检查是否是API请求
if (strpos($request_path, '/api/') === 0) {
    // 移除/api/前缀
    $api_path = substr($request_path, 5); // 移除 "/api/"
    
    // 解析API路径
    $path_parts = explode('/', trim($api_path, '/'));
    
    if (count($path_parts) >= 1) {
        $action = $path_parts[0];  // raw, meta, save
        $noteId = isset($path_parts[1]) ? $path_parts[1] : '';
        $password = isset($path_parts[2]) ? $path_parts[2] : null;
        
        // 验证ID格式
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $noteId)) {
            http_response_code(400);
            header('Content-Type: application/json');
            echo json_encode(['error' => '无效的笔记ID格式']);
            exit();
        }
        
        $noteFile = NOTES_DIR . 'note_' . $noteId . '.txt';
        $metaFile = NOTES_DIR . 'meta_' . $noteId . '.json';
        
        switch ($action) {
            // ========== 获取笔记内容 ==========
            case 'raw':
                if (file_exists($metaFile)) {
                    $meta = json_decode(file_get_contents($metaFile), true);
                    if (isset($meta['password'])) {
                        if ($password === null) {
                            http_response_code(403);
                            echo "需要密码";
                            exit();
                        }
                        
                        $inputHash = hash('sha256', $password);
                        if ($inputHash !== $meta['password']) {
                            http_response_code(403);
                            echo "密码错误";
                            exit();
                        }
                    }
                }
                
                $content = file_exists($noteFile) ? file_get_contents($noteFile) : '';
                header('Content-Type: text/plain; charset=utf-8');
                echo $content;
                exit();
                
            // ========== 获取笔记信息 ==========
            case 'meta':
                $meta = file_exists($metaFile) ? json_decode(file_get_contents($metaFile), true) : [];
                
                $response = [
                    'encrypted' => isset($meta['password']),
                    'format' => isset($meta['format']) ? $meta['format'] : 'txt',
                    'size' => file_exists($noteFile) ? filesize($noteFile) : 0,
                    'modified' => file_exists($noteFile) ? date('Y-m-d H:i:s', filemtime($noteFile)) : null
                ];
                
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode($response);
                exit();
                
            // ========== 保存笔记 ==========
            case 'save':
                // 只接受POST请求
                if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                    http_response_code(405);
                    header('Content-Type: application/json');
                    echo json_encode(['error' => '只接受POST请求']);
                    exit();
                }
                
                // 获取POST数据
                $input = file_get_contents('php://input');
                $data = json_decode($input, true);
                
                if ($data === null) {
                    http_response_code(400);
                    header('Content-Type: application/json');
                    echo json_encode(['error' => '无效的JSON数据']);
                    exit();
                }
                
                // 获取数据
                $content = isset($data['content']) ? $data['content'] : '';
                $format = isset($data['format']) ? $data['format'] : 'txt';
                $newPassword = isset($data['password']) ? $data['password'] : null;
                
                // 保存笔记内容
                if (file_put_contents($noteFile, $content) === false) {
                    http_response_code(500);
                    header('Content-Type: application/json');
                    echo json_encode(['error' => '保存笔记内容失败']);
                    exit();
                }
                
                // 构建元数据
                $metaData = ['format' => $format];
                if ($newPassword !== null && $newPassword !== '') {
                    $metaData['password'] = hash('sha256', trim($newPassword));
                }
                
                // 保存元数据
                if (file_put_contents($metaFile, json_encode($metaData, JSON_PRETTY_PRINT)) === false) {
                    http_response_code(500);
                    header('Content-Type: application/json');
                    echo json_encode(['error' => '保存元数据失败']);
                    exit();
                }
                
                // 返回成功响应
                header('Content-Type: application/json; charset=utf-8');
                echo json_encode([
                    'status' => 'success',
                    'message' => '保存成功',
                    'id' => $noteId,
                    'time' => date('Y-m-d H:i:s'),
                    'size' => strlen($content)
                ]);
                exit();
        }
    }
    
    // 如果没有匹配的API
    http_response_code(404);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'API路径不存在']);
    exit();
}

// ===================== 5. 显示前端页面 =====================
?>
<!DOCTYPE html>
<html lang="zh">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($currentId); ?> @ 我的笔记</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { 
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        /* 顶部状态栏 */
        .status-bar-top {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            background: rgba(255, 255, 255, 0.95);
            padding: 10px 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            z-index: 1000;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .status-left {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        .status-text {
            font-size: 14px;
            color: #333;
            padding: 4px 12px;
            background: #f0f0f0;
            border-radius: 20px;
            min-width: 120px;
        }
        .status-saving {
            background: #ffeb3b;
            color: #333;
        }
        .status-success {
            background: #4caf50;
            color: white;
        }
        .status-error {
            background: #f44336;
            color: white;
        }
        .note-id-badge {
            background: #667eea;
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-family: monospace;
            font-size: 14px;
        }
        
        .container {
            max-width: 1200px;
            margin: 80px auto 20px;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .header h1 {
            font-size: 32px;
            margin-bottom: 10px;
            font-weight: 600;
        }
        .header p {
            opacity: 0.9;
            font-size: 16px;
        }
        
        .toolbar {
            padding: 20px;
            background: #f8f9fa;
            border-bottom: 1px solid #e9ecef;
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
            align-items: center;
        }
        .toolbar input, .toolbar select, .toolbar button {
            padding: 10px 16px;
            border: 1px solid #ddd;
            border-radius: 8px;
            font-size: 14px;
            transition: all 0.3s ease;
        }
        .toolbar input {
            flex: 1;
            max-width: 300px;
            background: white;
        }
        .toolbar select {
            background: white;
            cursor: pointer;
        }
        .toolbar button {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            cursor: pointer;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .toolbar button:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        .toolbar button:active {
            transform: translateY(0);
        }
        
        .editor-container {
            display: flex;
            min-height: 600px;
            position: relative;
        }
        .line-numbers {
            width: 60px;
            background: #f8f9fa;
            padding: 20px 10px;
            text-align: right;
            font-family: 'Courier New', monospace;
            color: #6c757d;
            line-height: 1.6;
            user-select: none;
            border-right: 1px solid #e9ecef;
            font-size: 14px;
        }
        textarea {
            flex: 1;
            padding: 20px;
            border: none;
            font-family: 'Courier New', monospace;
            font-size: 15px;
            line-height: 1.6;
            resize: none;
            outline: none;
            background: white;
        }
        textarea:disabled {
            background: #f8f9fa;
            color: #6c757d;
        }
        
        .info-panel {
            padding: 20px;
            background: #f8f9fa;
            border-top: 1px solid #e9ecef;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 15px;
        }
        .char-count {
            font-size: 14px;
            color: #6c757d;
            background: white;
            padding: 6px 12px;
            border-radius: 20px;
            border: 1px solid #dee2e6;
        }
        .api-info {
            font-size: 13px;
            color: #6c757d;
        }
        .api-link {
            color: #667eea;
            text-decoration: none;
            font-family: monospace;
            background: white;
            padding: 6px 12px;
            border-radius: 6px;
            border: 1px solid #dee2e6;
        }
        .api-link:hover {
            text-decoration: underline;
        }
        
        /* 密码对话框 */
        .password-dialog {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            display: none;
            z-index: 1001;
            width: 90%;
            max-width: 400px;
        }
        .password-dialog h3 {
            margin-bottom: 15px;
            color: #333;
        }
        .password-dialog input {
            width: 100%;
            padding: 12px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            margin-bottom: 20px;
        }
        .password-dialog input:focus {
            border-color: #667eea;
            outline: none;
        }
        .dialog-buttons {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
        }
        .dialog-buttons button {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 500;
        }
        .dialog-buttons button:first-child {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        .dialog-buttons button:last-child {
            background: #f8f9fa;
            color: #6c757d;
        }
        
        .overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            display: none;
            z-index: 1000;
        }
        
        /* 响应式设计 */
        @media (max-width: 768px) {
            .container {
                margin: 60px auto 20px;
            }
            .header h1 {
                font-size: 24px;
            }
            .toolbar {
                flex-direction: column;
                align-items: stretch;
            }
            .toolbar input, .toolbar select, .toolbar button {
                width: 100%;
                max-width: none;
            }
            .info-panel {
                flex-direction: column;
                align-items: stretch;
            }
        }
    </style>
</head>
<body>
    <!-- 顶部状态栏 -->
    <div class="status-bar-top">
        <div class="status-left">
            <div class="note-id-badge">ID: <?php echo htmlspecialchars($currentId); ?></div>
            <div class="status-text" id="globalStatus">就绪</div>
        </div>
        <div style="font-size: 14px; color: #666;">
            <span id="currentTime"><?php echo date('H:i:s'); ?></span>
        </div>
    </div>
    
    <div class="container">
        <div class="header">
            <h1>📝 我的在线笔记</h1>
            <p>随时随地记录你的想法和代码片段</p>
        </div>
        
        <div class="toolbar">
            <input type="text" id="jumpInput" placeholder="输入笔记ID跳转..." value="<?php echo htmlspecialchars($currentId); ?>">
            <button onclick="jumpToNote()">🔗 跳转</button>
            
            <select id="formatSelect">
                <option value="txt">纯文本 (.txt)</option>
                <option value="json">JSON (.json)</option>
            </select>
            
            <button onclick="showPasswordDialog()">🔒 设置密码</button>
            <button onclick="newNote()">📄 新建笔记</button>
            <button onclick="shareNote()">📤 分享链接</button>
            <button onclick="toggleFullscreen()">⛶ 全屏</button>
        </div>
        
        <div class="editor-container">
            <div class="line-numbers" id="lineNumbers">1</div>
            <textarea id="editor" placeholder="开始输入你的笔记内容...支持自动保存，修改后1秒自动保存"></textarea>
        </div>
        
        <div class="info-panel">
            <div class="char-count">
                字符数: <span id="charCount">0</span> | 
                行数: <span id="lineCount">1</span>
            </div>
            <div class="api-info">
                API地址: <a href="#" id="rawUrl" class="api-link" target="_blank">正在生成...</a>
            </div>
        </div>
    </div>
    
    <!-- 密码对话框 -->
    <div class="overlay" id="overlay" onclick="hidePasswordDialog()"></div>
    <div class="password-dialog" id="passwordDialog">
        <h3>🔒 设置密码保护</h3>
        <p>输入密码（留空则移除密码保护）：</p>
        <input type="password" id="passwordInput" placeholder="输入密码...">
        <div class="dialog-buttons">
            <button onclick="savePassword()">确定</button>
            <button onclick="hidePasswordDialog()">取消</button>
        </div>
    </div>
    
    <script>
        // ===================== 前端JavaScript =====================
        const currentId = '<?php echo $currentId; ?>';
        let isAuthorized = false;
        let saveTimer = null;
        let isSaving = false;
        
        // 获取当前页面的基础URL
        const baseUrl = window.location.pathname;
        
        // 获取DOM元素
        const editor = document.getElementById('editor');
        const lineNumbers = document.getElementById('lineNumbers');
        const formatSelect = document.getElementById('formatSelect');
        const charCount = document.getElementById('charCount');
        const lineCount = document.getElementById('lineCount');
        const rawUrl = document.getElementById('rawUrl');
        const globalStatus = document.getElementById('globalStatus');
        const passwordDialog = document.getElementById('passwordDialog');
        const overlay = document.getElementById('overlay');
        const currentTime = document.getElementById('currentTime');
        const jumpInput = document.getElementById('jumpInput');
        
        // ========== 页面加载时执行 ==========
        document.addEventListener('DOMContentLoaded', function() {
            // 更新API链接
            updateApiUrl();
            
            // 加载笔记
            loadNote();
            
            // 绑定事件
            editor.addEventListener('input', function() {
                updateLineNumbers();
                updateCharCount();
                autoSave();
            });
            
            editor.addEventListener('scroll', function() {
                lineNumbers.scrollTop = editor.scrollTop;
            });
            
            formatSelect.addEventListener('change', function() {
                autoSave();
            });
            
            // 更新当前时间
            updateCurrentTime();
            setInterval(updateCurrentTime, 1000);
            
            // 跳转输入框按Enter键跳转
            jumpInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    jumpToNote();
                }
            });
            
            console.log('📝 页面已加载');
            console.log('当前笔记ID:', currentId);
            console.log('基础URL:', baseUrl);
        });
        
        // ========== 更新当前时间 ==========
        function updateCurrentTime() {
            const now = new Date();
            currentTime.textContent = now.toLocaleTimeString('zh-CN', { 
                hour12: false,
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
        }
        
        // ========== 更新行号 ==========
        function updateLineNumbers() {
            const lines = editor.value.split('\n').length;
            let numbers = '';
            for (let i = 1; i <= lines; i++) {
                numbers += i + '\n';
            }
            lineNumbers.textContent = numbers;
            lineCount.textContent = lines;
        }
        
        // ========== 更新字符计数 ==========
        function updateCharCount() {
            const count = editor.value.length;
            charCount.textContent = count;
        }
        
        // ========== 更新API链接 ==========
        function updateApiUrl() {
            // 使用完整路径，确保API调用正确
            const url = window.location.origin + baseUrl + 'api/raw/' + encodeURIComponent(currentId);
            rawUrl.href = url;
            rawUrl.textContent = url;
        }
        
        // ========== 获取API基础URL ==========
        function getApiBaseUrl() {
            // 返回当前脚本的基础URL
            return window.location.origin + baseUrl + 'api/';
        }
        
        // ========== 设置状态消息 ==========
        function setStatus(message, type = 'info') {
            globalStatus.textContent = message;
            globalStatus.className = 'status-text';
            
            switch(type) {
                case 'saving':
                    globalStatus.classList.add('status-saving');
                    break;
                case 'success':
                    globalStatus.classList.add('status-success');
                    break;
                case 'error':
                    globalStatus.classList.add('status-error');
                    break;
            }
            
            console.log('状态:', message, '类型:', type);
        }
        
        // ========== 加载笔记 ==========
        async function loadNote() {
            try {
                setStatus('正在加载...', 'saving');
                
                const apiBase = getApiBaseUrl();
                console.log('加载笔记，API地址:', apiBase + 'meta/' + encodeURIComponent(currentId));
                
                // 获取笔记信息
                const metaRes = await fetch(apiBase + 'meta/' + encodeURIComponent(currentId));
                console.log('meta响应状态:', metaRes.status);
                
                if (!metaRes.ok) {
                    // 笔记不存在，创建新笔记
                    setStatus('创建新笔记', 'success');
                    editor.value = '# 新笔记\n\n欢迎使用在线笔记！\n\n开始记录你的想法吧...';
                    isAuthorized = true;
                    updateLineNumbers();
                    updateCharCount();
                    return;
                }
                
                const meta = await metaRes.json();
                console.log('元数据:', meta);
                
                // 检查是否需要密码
                if (meta.encrypted) {
                    const password = prompt('🔒 此笔记已加密，请输入密码：');
                    if (!password) {
                        editor.value = '需要密码才能查看此笔记';
                        editor.disabled = true;
                        setStatus('需要密码', 'error');
                        return;
                    }
                    
                    // 使用密码获取内容
                    const contentRes = await fetch(apiBase + 'raw/' + encodeURIComponent(currentId) + '/' + encodeURIComponent(password));
                    console.log('加密内容响应状态:', contentRes.status);
                    
                    if (contentRes.ok) {
                        editor.value = await contentRes.text();
                        isAuthorized = true;
                        setStatus('已加载（加密）', 'success');
                    } else {
                        editor.value = '密码错误';
                        editor.disabled = true;
                        setStatus('密码错误', 'error');
                        return;
                    }
                } else {
                    // 获取内容（无密码）
                    const contentRes = await fetch(apiBase + 'raw/' + encodeURIComponent(currentId));
                    console.log('内容响应状态:', contentRes.status);
                    
                    if (contentRes.ok) {
                        editor.value = await contentRes.text();
                        isAuthorized = true;
                        setStatus('已加载', 'success');
                    } else {
                        editor.value = '';
                        isAuthorized = true;
                        setStatus('空笔记', 'success');
                    }
                }
                
                // 设置格式
                formatSelect.value = meta.format || 'txt';
                
                // 更新UI
                updateLineNumbers();
                updateCharCount();
                
            } catch (error) {
                console.error('加载笔记失败:', error);
                setStatus('加载失败: ' + error.message, 'error');
                editor.value = '加载失败：' + error.message;
            }
        }
        
        // ========== 自动保存 ==========
        function autoSave() {
            if (!isAuthorized || isSaving) return;
            
            clearTimeout(saveTimer);
            saveTimer = setTimeout(() => {
                saveNote();
            }, 1000);
        }
        
        // ========== 保存笔记 ==========
        async function saveNote(password = null) {
            if (!isAuthorized) {
                setStatus('未授权，无法保存', 'error');
                return;
            }
            
            if (isSaving) {
                console.log('正在保存中，跳过...');
                return;
            }
            
            try {
                isSaving = true;
                setStatus('正在保存...', 'saving');
                
                const apiBase = getApiBaseUrl();
                const saveUrl = apiBase + 'save/' + encodeURIComponent(currentId);
                console.log('保存URL:', saveUrl);
                
                const data = {
                    content: editor.value,
                    format: formatSelect.value
                };
                
                // 如果提供了密码，添加到数据中
                if (password !== null) {
                    data.password = password;
                }
                
                console.log('保存数据:', data);
                
                const response = await fetch(saveUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                });
                
                console.log('保存响应状态:', response.status);
                
                if (response.ok) {
                    const result = await response.json();
                    console.log('保存成功:', result);
                    setStatus('保存成功 ' + new Date().toLocaleTimeString().substr(0,5), 'success');
                    
                    // 更新API链接（如果设置了密码）
                    if (password !== null) {
                        updateApiUrl();
                    }
                } else {
                    try {
                        const errorResult = await response.json();
                        console.error('保存失败:', errorResult);
                        setStatus('保存失败: ' + (errorResult.error || response.statusText), 'error');
                    } catch (e) {
                        console.error('保存失败，无法解析错误信息:', e);
                        setStatus('保存失败: ' + response.statusText, 'error');
                    }
                }
                
            } catch (error) {
                console.error('保存请求失败:', error);
                setStatus('保存失败: ' + error.message, 'error');
            } finally {
                isSaving = false;
            }
        }
        
        // ========== 跳转到其他笔记 ==========
        function jumpToNote() {
            const newId = jumpInput.value.trim();
            
            if (newId) {
                if (newId === currentId) {
                    setStatus('已经是当前笔记', 'info');
                    return;
                }
                
                // 构建跳转URL
                let newUrl = baseUrl;
                if (newUrl.includes('?')) {
                    newUrl = newUrl.split('?')[0];
                }
                newUrl += '?id=' + encodeURIComponent(newId);
                
                console.log('跳转到:', newUrl);
                window.location.href = newUrl;
            } else {
                setStatus('请输入笔记ID', 'error');
            }
        }
        
        // ========== 新建笔记 ==========
        function newNote() {
            const randomId = 'note-' + Date.now();
            let newUrl = baseUrl;
            if (newUrl.includes('?')) {
                newUrl = newUrl.split('?')[0];
            }
            newUrl += '?id=' + randomId;
            window.location.href = newUrl;
        }
        
        // ========== 分享链接 ==========
        function shareNote() {
            const url = window.location.href;
            console.log('分享链接:', url);
            
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(url).then(() => {
                    setStatus('链接已复制到剪贴板！', 'success');
                    setTimeout(() => setStatus('就绪'), 2000);
                }).catch(err => {
                    console.error('复制失败:', err);
                    prompt('请复制以下链接：', url);
                });
            } else {
                prompt('请复制以下链接：', url);
            }
        }
        
        // ========== 全屏切换 ==========
        function toggleFullscreen() {
            const elem = document.documentElement;
            if (!document.fullscreenElement) {
                if (elem.requestFullscreen) {
                    elem.requestFullscreen();
                } else if (elem.webkitRequestFullscreen) {
                    elem.webkitRequestFullscreen();
                } else if (elem.msRequestFullscreen) {
                    elem.msRequestFullscreen();
                }
                setStatus('全屏模式已开启', 'success');
            } else {
                if (document.exitFullscreen) {
                    document.exitFullscreen();
                } else if (document.webkitExitFullscreen) {
                    document.webkitExitFullscreen();
                } else if (document.msExitFullscreen) {
                    document.msExitFullscreen();
                }
                setStatus('全屏模式已关闭', 'success');
            }
        }
        
        // ========== 密码对话框 ==========
        function showPasswordDialog() {
            if (!isAuthorized) {
                alert('请先加载笔记才能设置密码');
                return;
            }
            passwordDialog.style.display = 'block';
            overlay.style.display = 'block';
        }
        
        function hidePasswordDialog() {
            passwordDialog.style.display = 'none';
            overlay.style.display = 'none';
        }
        
        async function savePassword() {
            const password = document.getElementById('passwordInput').value;
            await saveNote(password);
            hidePasswordDialog();
            setStatus('密码设置成功！', 'success');
            setTimeout(() => setStatus('就绪'), 2000);
        }
        
        // ========== 键盘快捷键 ==========
        document.addEventListener('keydown', function(e) {
            // Ctrl+S 保存
            if (e.ctrlKey && e.key === 's') {
                e.preventDefault();
                saveNote();
            }
            
            // Ctrl+F 搜索
            if (e.ctrlKey && e.key === 'f') {
                e.preventDefault();
                const searchText = prompt('搜索内容：');
                if (searchText) {
                    const index = editor.value.indexOf(searchText);
                    if (index !== -1) {
                        editor.focus();
                        editor.setSelectionRange(index, index + searchText.length);
                    } else {
                        alert('未找到 "' + searchText + '"');
                    }
                }
            }
            
            // Ctrl+G 跳转
            if (e.ctrlKey && e.key === 'g') {
                e.preventDefault();
                jumpInput.focus();
            }
        });
        
        // 导出函数供全局使用
        window.saveNote = saveNote;
        window.jumpToNote = jumpToNote;
    </script>
</body>
</html>