<?php
// index.php

// --- PHP 后端 API 处理 ---

// 定义笔记存储目录
define('NOTES_DIR', __DIR__ . '/notes/');

// 确保存储目录存在且可写
if (!is_dir(NOTES_DIR)) {
    if (!mkdir(NOTES_DIR, 0755, true)) {
        die('无法创建笔记存储目录: ' . NOTES_DIR);
    }
}
if (!is_writable(NOTES_DIR)) {
    die('笔记存储目录不可写: ' . NOTES_DIR);
}

// --- CORS 设置 ---
function setCORSHeaders($request) {
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    if ($origin && preg_match('/\.008890\.xyz$/', parse_url($origin, PHP_URL_HOST))) {
        header("Access-Control-Allow-Origin: " . $origin);
        header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
        header("Access-Control-Allow-Headers: Content-Type");
        return true;
    }
    // 如果 Origin 不匹配，不设置 CORS 头
    return false;
}

// 处理 OPTIONS 预检请求
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    $allowed = setCORSHeaders($_SERVER);
    if ($allowed) {
        http_response_code(204);
    } else {
        http_response_code(403);
    }
    exit();
}

// 获取请求路径
$request_uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$script_name = $_SERVER['SCRIPT_NAME']; // e.g., /paste/index.php

// 如果路径以脚本名开头，去掉它 (用于处理直接访问 index.php 的情况)
if (substr($request_uri, 0, strlen($script_name)) === $script_name) {
    $request_uri = substr($request_uri, strlen($script_name));
}

// --- 修正：解析 API 路径 ---
$api_path = $request_uri; // 默认为整个请求路径
$script_dir = dirname($script_name); // 获取脚本所在目录，例如 "/paste"
// 如果请求路径以脚本目录开头，则去掉目录前缀
if (substr($request_uri, 0, strlen($script_dir)) === $script_dir) {
    $api_path = substr($request_uri, strlen($script_dir));
}

// 如果路径为空或为 / (相对于脚本目录)，则显示前端页面
if ($api_path === '' || $api_path === '/') {
    // 输出前端 HTML (这部分来自之前的 HTML_PAGE)
    $currentId = $_GET['id'] ?? 'default'; // 可以通过 ?id=xxx 传递 ID
    if (empty($currentId)) $currentId = 'default';
    $page_title = htmlspecialchars($currentId) . ' @ Note';
    // 注意：这里替换了前端 JS 中的 API URL 前缀
    echo str_replace(
        ['<title>default @ Note</title>', 'location.origin + \'/paste/api/', 'fetch(\'/paste/api/'],
        ['<title>' . $page_title . '</title>', 'location.origin + \'/paste/api/', 'fetch(\'/paste/api/'],
        getHTMLPage()
    );
    exit();
}

// API 路由处理 (现在需要匹配相对于脚本目录的 /api/... )
if (preg_match('#^/api/(raw|meta|save)/([^/]+)(?:/([^/]+))?$#', $api_path, $matches)) {
    $api_action = $matches[1]; // 'raw', 'meta', 'save'
    $id = $matches[2];
    $password = $matches[3] ?? null; // 仅 /api/raw/xxx/yyy 中的 yyy

    // 验证 ID 格式（简单白名单，防止路径遍历）
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $id)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid ID format']);
        exit();
    }

    // 设置 CORS 头（如果允许）
    $cors_allowed = setCORSHeaders($_SERVER);

    $note_file = NOTES_DIR . 'note_' . $id . '.txt';
    $meta_file = NOTES_DIR . 'meta_' . $id . '.json';

    switch ($api_action) {
        case 'raw':
            // 获取笔记内容
            if (file_exists($meta_file)) {
                $meta = json_decode(file_get_contents($meta_file), true);
                $stored_password_hash = $meta['password'] ?? null;

                if ($stored_password_hash) {
                    if ($password === null) {
                        http_response_code(403);
                        echo "Password required";
                        break;
                    }
                    $input_hash = hash('sha256', $password);
                    if ($input_hash !== $stored_password_hash) {
                        http_response_code(403);
                        echo "Password mismatch";
                        break;
                    }
                }
            }
            $content = file_exists($note_file) ? file_get_contents($note_file) : '';
            header('Content-Type: text/plain; charset=utf-8');
            echo $content;
            break;

        case 'meta':
            // 获取笔记元数据
            $meta = file_exists($meta_file) ? json_decode(file_get_contents($meta_file), true) : [];
            $response_data = [
                'encrypted' => isset($meta['password']),
                'format' => $meta['format'] ?? 'txt'
            ];
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode($response_data);
            break;

        case 'save':
            // 保存笔记内容 (仅接受 POST)
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                http_response_code(405);
                echo json_encode(['error' => 'Method not allowed']);
                break;
            }
            $input = json_decode(file_get_contents('php://input'), true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON']);
                break;
            }

            $content = $input['content'] ?? '';
            $format = $input['format'] ?? 'txt';
            $new_password = $input['password']; // Can be null/string

            // 保存内容
            file_put_contents($note_file, $content);

            // 保存元数据
            $meta = ['format' => $format];
            if ($new_password !== null && $new_password !== '') {
                $meta['password'] = hash('sha256', trim($new_password));
            }
            file_put_contents($meta_file, json_encode($meta));

            header('Content-Type: application/json; charset=utf-8');
            echo json_encode(['status' => 'OK']);
            break;

        default:
            http_response_code(404);
            echo json_encode(['error' => 'API endpoint not found']);
    }
    exit(); // API 处理完毕，退出
}

// 如果没有匹配到任何路由，则返回 404
http_response_code(404);
echo "Not Found";

// --- 函数定义 ---

function getHTMLPage() {
    // 这里粘贴之前 HTML_PAGE 的内容
    // 注意：API URL 路径 *必须* 在 PHP 中动态替换，因为 JS 代码需要知道正确的前缀
    // 我们在输出页面时，会将 /paste/api/ 保持不变，因为 JS 代码中已经写好了
    return '<!DOCTYPE html>
<html lang="zh">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>default @ Note</title> <!-- 标题将根据笔记 ID 动态修改 -->
  <style>
    * { box-sizing: border-box; }
    html, body {
      height: 100%;
      margin: 0;
      padding: 0;
      font-family: -apple-system, BlinkMacSystemFont, sans-serif;
      background: #fff;
      color: #333;
    }
    #app {
      display: flex;
      flex-direction: column;
      min-height: 100vh;
      padding: 12px;
    }
    .toolbar {
      margin: 0 0 10px 0;
      display: flex;
      flex-wrap: wrap;
      gap: 8px;
      align-items: center;
    }
    select, button {
      padding: 6px 10px;
      font-size: 14px;
    }
    .api-url {
      margin-top: 6px;
      word-break: break-all;
      font-size: 13px;
    }
    .api-url a {
      color: #1a73e8;
      text-decoration: none;
    }
    .api-url a:hover {
      text-decoration: underline;
    }
    .status-bar {
      font-size: 13px;
      color: #666;
      margin-top: 6px;
      min-height: 18px;
    }
    #editor-container {
      position: relative;
      flex: 1;
      display: flex;
      min-height: 200px;
      border: 1px solid #ccc;
    }
    #line-numbers {
      width: 48px;
      padding: 10px 6px;
      text-align: right;
      background: #f6f6f6;
      color: #999;
      font-family: monospace;
      font-size: 15px;
      line-height: 1.4;
      user-select: none;
      white-space: pre;
      overflow: hidden;
    }
    #editor {
      flex: 1;
      font-family: monospace;
      font-size: 15px;
      padding: 10px;
      resize: none;
      line-height: 1.4;
      tab-size: 2;
      border: none;
      outline: none;
      overflow: auto;
    }
    #json-preview {
      margin-top: 10px;
      padding: 12px;
      background: #f8f8f8;
      border: 1px solid #ddd;
      border-radius: 4px;
      font-family: monospace;
      font-size: 14px;
      line-height: 1.4;
      white-space: pre;
      overflow: auto;
      display: none;
    }
    /* JSON 高亮颜色 */
    .json-key { color: #d73a49; }       /* 橙红：键名 */
    .json-string { color: #032f62; }    /* 深蓝：字符串 */
    .json-number { color: #b623b6; }    /* 紫色：数字 */
    .json-boolean { color: #005cc5; }   /* 蓝色：true/false */
    .json-null { color: #005cc5; }      /* 蓝色：null */
    .json-bracket { color: #24292e; }   /* 黑色：{}[] */
    .password-dialog {
      position: fixed;
      top: 20%;
      left: 10%;
      right: 10%;
      background: white;
      border: 1px solid #ccc;
      padding: 16px;
      z-index: 1000;
      display: none;
    }
    .overlay {
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0,0,0,0.3);
      z-index: 999;
      display: none;
    }
    .locked {
      background: #f9f9f9;
      color: #999;
    }
  </style>
</head>
<body>
  <div id="app">
    <div class="toolbar">
      <select id="format" disabled>
        <option value="txt">txt</option>
        <option value="json">json</option>
      </select>
      <button id="lockBtn" onclick="showPasswordDialog()" disabled>🔒 加密模式</button>
      <button onclick="location.hash = \'note-\' + Date.now(); location.reload();">📂 新建</button>
      <div class="api-url" id="apiUrl"></div>
    </div>
    <div id="editor-container">
      <div id="line-numbers"></div>
      <textarea id="editor" placeholder="加载中..." disabled></textarea>
    </div>
    <pre id="json-preview"></pre>
    <div class="status-bar" id="statusBar">加载中...</div>
    <div class="overlay" id="overlay" onclick="closePasswordDialog()"></div>
    <div class="password-dialog" id="pwdDialog">
      <h3>加密模式</h3>
      <p>输入新密码（留空可删除密码）：</p>
      <input type="password" id="dialogPassword" placeholder="留空则删除密码">
      <div style="margin-top:10px;">
        <button onclick="savePassword()">修改</button>
        <button onclick="closePasswordDialog()">关闭</button>
      </div>
    </div>
  </div>
  <script>
    // --- 修改：获取当前 ID ---
    let currentId = location.hash.slice(1) || \'default\';
    if (!currentId) currentId = \'default\';
    // 修改网页标题
    document.title = currentId + \' @ Note\';
    const editor = document.getElementById(\'editor\');
    const lineNumbers = document.getElementById(\'line-numbers\');
    const formatSel = document.getElementById(\'format\');
    const lockBtn = document.getElementById(\'lockBtn\');
    const apiUrlEl = document.getElementById(\'apiUrl\');
    const statusBar = document.getElementById(\'statusBar\');
    const jsonPreview = document.getElementById(\'json-preview\');
    // --- 修改：API URL (现在包含 /paste/ 前缀) ---
    const apiUrl = location.origin + \'/paste/api/raw/\' + currentId;
    apiUrlEl.innerHTML = \'<a href="\' + apiUrl + \'" target="_blank">\' + apiUrl + \'</a>\';
    let isAuthorized = false;
    let saveTimer = null;
    // 行号
    function updateLineNumbers() {
      const lines = editor.value.split(\'\\n\').length;
      let numbers = \'\';
      for (let i = 1; i <= lines; i++) {
        numbers += i.toString().padStart(2, \' \') + \'\\n\';
      }
      lineNumbers.textContent = numbers;
    }
    // JSON 高亮函数
    function syntaxHighlight(json) {
      if (typeof json !== \'string\') {
        json = JSON.stringify(json, undefined, 2);
      }
      json = json.replace(/&/g, \'&amp;\').replace(/</g, \'&lt;\').replace(/>/g, \'&gt;\');
      return json.replace(/("(\\\\u[a-zA-Z0-9]{4}|\\\\[^u]|[^\\"])*"(\\s*:)?|\\b(true|false|null)\\b|-?\\\\d+(?:\\.\\\\d*)?(?:[eE][+\\\\-]?\\\\d+)?)/g, function (match) {
        let cls = \'json-bracket\';
        if (/^"/.test(match)) {
          if (/:$/.test(match)) {
            cls = \'json-key\';
          } else {
            cls = \'json-string\';
          }
        } else if (/true|false/.test(match)) {
          cls = \'json-boolean\';
        } else if (/null/.test(match)) {
          cls = \'json-null\';
        } else if (/^-?\\\\d/.test(match)) {
          cls = \'json-number\';
        }
        return \'<span class="\' + cls + \'">\' + match + \'</span>\';
      });
    }
    function updateJsonPreview() {
      if (formatSel.value === \'json\') {
        try {
          const obj = JSON.parse(editor.value);
          jsonPreview.innerHTML = syntaxHighlight(obj);
          jsonPreview.style.display = \'block\';
        } catch (e) {
          jsonPreview.textContent = \'❌ JSON 格式错误\';
          jsonPreview.style.color = \'#d32f2f\';
          jsonPreview.style.display = \'block\';
        }
      } else {
        jsonPreview.style.display = \'none\';
      }
    }
    function setStatus(text, isError = false) {
      statusBar.textContent = text;
      statusBar.style.color = isError ? \'#d32f2f\' : \'#666\';
    }
    // --- 修改：fetchWithTimeout 函数，确保 API 路径正确 ---
    async function fetchWithTimeout(url, options = {}, timeout = 5000) {
      // 确保 API 请求使用 /paste/api/ 前缀
      // 如果 URL 是相对路径（以 / 开头）且不包含 /paste/，则添加前缀
      if (!url.startsWith(\'http\') && url.startsWith(\'/\') && !url.includes(\'/paste/\')) {
          url = location.origin + \'/paste\' + url;
      }
      // 如果 URL 已经包含 /paste/，则直接使用
      const controller = new AbortController();
      const id = setTimeout(() => controller.abort(), timeout);
      try {
        const response = await fetch(url, { ...options, signal: controller.signal });
        clearTimeout(id);
        return response;
      } catch (e) {
        clearTimeout(id);
        throw e;
      }
    }
    async function tryAutoLogin() {
      const saved = localStorage.getItem(\'note_auth\');
      if (!saved) return null;
      try {
        const auths = JSON.parse(saved);
        const record = auths[currentId];
        if (record && record.password) {
          try {
            // --- 修改：API URL (现在包含 /paste/ 前缀) ---
            const contentRes = await fetchWithTimeout(
              \'/paste/api/raw/\' + encodeURIComponent(currentId) + \'/\' + encodeURIComponent(record.password)
            );
            if (contentRes.ok) {
              return { password: record.password, content: await contentRes.text() };
            } else {
              delete auths[currentId];
              localStorage.setItem(\'note_auth\', JSON.stringify(auths));
            }
          } catch (e) {
            setStatus(\'自动登录超时\', true);
          }
        }
      } catch (e) {
        console.warn(\'Auto-login failed\', e);
      }
      return null;
    }
    function saveAuth(password) {
      try {
        const auths = JSON.parse(localStorage.getItem(\'note_auth\') || \'{}\');
        auths[currentId] = { password };
        localStorage.setItem(\'note_auth\', JSON.stringify(auths));
      } catch (e) {
        console.warn(\'Failed to save auth\', e);
      }
    }
    function clearAuth() {
      try {
        const auths = JSON.parse(localStorage.getItem(\'note_auth\') || \'{}\');
        delete auths[currentId];
        localStorage.setItem(\'note_auth\', JSON.stringify(auths));
      } catch (e) {
        console.warn(\'Failed to clear auth\', e);
      }
    }
    async function loadNote() {
      // --- 修改：API URL (现在包含 /paste/ 前缀) ---
      const metaPromise = fetchWithTimeout(\'/paste/api/meta/\' + encodeURIComponent(currentId));
      let isEncrypted = false;
      let format = \'txt\'; // 默认 txt
      try {
        const metaRes = await metaPromise;
        if (metaRes.ok) {
          const meta = await metaRes.json();
          isEncrypted = !!meta.encrypted;
          format = meta.format || \'txt\';
        }
      } catch (e) {
        setStatus(\'加载元信息失败\', true);
        editor.value = \'加载失败\';
        return;
      }
      if (isEncrypted) {
        const auto = await tryAutoLogin();
        if (auto) {
          editor.value = auto.content;
          isAuthorized = true;
          enableEditing();
          formatSel.value = format;
          setStatus(\'已加载（自动登录）\');
          updateLineNumbers();
          updateJsonPreview();
          return;
        }
        const pwd = prompt(\'此笔记已加密，请输入密码：\');
        if (pwd === null) {
          editor.value = \'未提供密码\';
          editor.classList.add(\'locked\');
          setStatus(\'未提供密码\');
          return;
        }
        try {
          // --- 修改：API URL (现在包含 /paste/ 前缀) ---
          const contentRes = await fetchWithTimeout(
            \'/paste/api/raw/\' + encodeURIComponent(currentId) + \'/\' + encodeURIComponent(pwd)
          );
          if (contentRes.ok) {
            editor.value = await contentRes.text();
            isAuthorized = true;
            enableEditing();
            formatSel.value = format;
            saveAuth(pwd);
            setStatus(\'已加载\');
          } else {
            alert(\'❌ 密码错误\');
            editor.value = \'\';
            editor.classList.add(\'locked\');
            setStatus(\'密码错误\', true);
          }
        } catch (e) {
          setStatus(\'请求超时\', true);
        }
      } else {
        try {
          // --- 修改：API URL (现在包含 /paste/ 前缀) ---
          const contentRes = await fetchWithTimeout(\'/paste/api/raw/\' + encodeURIComponent(currentId));
          editor.value = contentRes.ok ? await contentRes.text() : \'\';
          isAuthorized = true;
          enableEditing();
          formatSel.value = format;
          clearAuth();
          setStatus(\'已加载\');
        } catch (e) {
          setStatus(\'加载内容失败\', true);
          editor.value = \'加载失败\';
        }
      }
      updateLineNumbers();
      updateJsonPreview();
    }
    function enableEditing() {
      editor.disabled = false;
      formatSel.disabled = false;
      lockBtn.disabled = false;
      editor.classList.remove(\'locked\');
    }
    function showPasswordDialog() {
      if (!isAuthorized) return;
      document.getElementById(\'overlay\').style.display = \'block\';
      document.getElementById(\'pwdDialog\').style.display = \'block\';
      document.getElementById(\'dialogPassword\').value = \'\';
    }
    function closePasswordDialog() {
      document.getElementById(\'overlay\').style.display = \'none\';
      document.getElementById(\'pwdDialog\').style.display = \'none\';
    }
    async function savePassword() {
      if (!isAuthorized) return;
      const pwd = document.getElementById(\'dialogPassword\').value.trim() || null;
      await saveNote(pwd);
      closePasswordDialog();
      if (pwd) {
        // --- 修改：API URL (现在包含 /paste/ 前缀) ---
        const newUrl = location.origin + \'/paste/api/raw/\' + currentId + \'/\' + pwd;
        apiUrlEl.innerHTML = \'<a href="\' + newUrl + \'" target="_blank">\' + newUrl + \'</a>\';
        saveAuth(pwd);
      } else {
        // --- 修改：API URL (现在包含 /paste/ 前缀) ---
        const newUrl = location.origin + \'/paste/api/raw/\' + currentId;
        apiUrlEl.innerHTML = \'<a href="\' + newUrl + \'" target="_blank">\' + newUrl + \'</a>\';
        clearAuth();
      }
    }
    async function saveNote(password) {
      if (!isAuthorized) return;
      setStatus(\'正在保存...\');
      try {
        const payload = {
          content: editor.value,
          format: formatSel.value,
          password: password
        };
        // --- 修改：API URL (现在包含 /paste/ 前缀) ---
        const res = await fetchWithTimeout(\'/paste/api/save/\' + encodeURIComponent(currentId), {
          method: \'POST\',
          headers: { \'Content-Type\': \'application/json\' },
          body: JSON.stringify(payload)
        });
        if (res.ok) {
          setStatus(\'已保存\');
        } else {
          setStatus(\'保存失败：\' + res.status, true);
        }
      } catch (e) {
        setStatus(\'保存超时\', true);
      }
    }
    function autoSaveContent() {
      if (!isAuthorized) return;
      clearTimeout(saveTimer);
      saveTimer = setTimeout(() => saveNote(null), 1000);
    }
    editor.addEventListener(\'input\', () => {
      autoSaveContent();
      updateLineNumbers();
      updateJsonPreview();
    });
    formatSel.addEventListener(\'change\', () => {
      autoSaveContent();
      updateJsonPreview();
    });
    editor.addEventListener(\'scroll\', () => {
      lineNumbers.scrollTop = editor.scrollTop;
    });
    loadNote();
  </script>
</body>
</html>';
}
?>